#!/usr/bin/env python3
"""
DTR2 ALP File Converter - Convert between ALP and DXF/OBJ formats

This tool allows you to:
1. Convert ALP files from DTR2 to DXF format for editing in Rhino 3D
2. Convert ALP files from DTR2 to OBJ format for editing in various 3D software
3. Convert modified DXF/OBJ files back to ALP format for use in the game

The converter preserves all necessary data including texture paths, face indices,
and additional metadata to ensure proper reconstruction of the ALP format.
"""

import os
import tkinter as tk
from tkinter import filedialog, messagebox
from tkinter import ttk
import ezdxf
import re
import traceback  # Added for better error reporting

###############################################################################
#                          ALP PARSING FUNCTIONS                              #
###############################################################################

def parse_alp_file(filename):
    """Parse an ALP file and return the data structure"""
    try:
        with open(filename, 'r') as f:
            lines = f.read().splitlines()

        sections = []
        current_section = None
        i = 0

        while i < len(lines):
            if i >= len(lines):
                break
                
            line = lines[i].strip()
            
            # Skip "num meshes" header line - added to handle this case
            if line.startswith("num meshes"):
                i += 1
                continue
                
            if line.startswith("num verts"):
                if current_section is not None:
                    sections.append(current_section)
                current_section = {"verts": [], "tverts": [], "objects": []}
                try:
                    parts = line.split()
                    num_verts = int(parts[2]) if len(parts) > 2 else 0
                except:
                    num_verts = 0
                i += 1
                
                # Safety check to avoid going out of bounds
                vert_count = 0
                while vert_count < num_verts and i < len(lines):
                    v_line = lines[i].strip()
                    if not v_line:  # Skip empty lines
                        i += 1
                        continue
                    
                    if v_line.startswith("num"):  # We've hit a new section
                        break
                        
                    try:
                        coords = list(map(float, v_line.split()))
                        if len(coords) >= 3:  # Ensure we have at least x, y, z
                            current_section["verts"].append(coords[:3])  # Take only the first 3 values
                            vert_count += 1
                    except ValueError:
                        # Skip lines that can't be converted to floats
                        pass
                    i += 1

            elif line.startswith("num tverts"):
                try:
                    parts = line.split()
                    num_tverts = int(parts[2]) if len(parts) > 2 else 0
                except:
                    num_tverts = 0
                i += 1
                
                # Safety check to avoid going out of bounds
                tvert_count = 0
                while tvert_count < num_tverts and i < len(lines):
                    t_line = lines[i].strip()
                    if not t_line:  # Skip empty lines
                        i += 1
                        continue
                        
                    if t_line.startswith("num"):  # We've hit a new section
                        break
                        
                    try:
                        coords = list(map(float, t_line.split()))
                        if len(coords) >= 2:  # Ensure we have at least u, v
                            current_section["tverts"].append(coords[:2])  # Take only the first 2 values
                            tvert_count += 1
                    except ValueError:
                        # Skip lines that can't be converted to floats
                        pass
                    i += 1

            elif line.startswith("num objects"):
                if current_section is None:
                    current_section = {"verts": [], "tverts": [], "objects": []}
                    
                try:
                    parts = line.split()
                    num_objects = int(parts[2]) if len(parts) > 2 else 0
                except:
                    num_objects = 0
                i += 1
                
                for _ in range(num_objects):
                    if i >= len(lines):
                        break
                        
                    texture_line = lines[i].strip()
                    texture = ""
                    if texture_line.startswith("texture"):
                        texture = texture_line[len("texture"):].strip()
                    i += 1

                    num_faces = 0
                    if i < len(lines) and lines[i].strip().startswith("num faces"):
                        try:
                            parts = lines[i].split()
                            num_faces = int(parts[2]) if len(parts) > 2 else 0
                        except:
                            pass
                        i += 1

                    faces = []
                    for _ in range(num_faces):
                        if i >= len(lines):
                            break
                            
                        face_line = lines[i].strip().split()
                        
                        # Initialize with safe defaults
                        v_indices = []
                        vt_indices = []
                        extra = []
                        
                        # Extract vertex indices (safer approach)
                        v_count = min(3, len(face_line))  # Take up to 3 indices
                        for j in range(v_count):
                            try:
                                v_indices.append(int(face_line[j]))
                            except (ValueError, IndexError):
                                # Use a placeholder or default if conversion fails
                                v_indices.append(0)
                        
                        # Ensure we always have 3 vertex indices
                        while len(v_indices) < 3:
                            v_indices.append(0)
                        
                        # Extract texture indices (safer approach)
                        if len(face_line) > 3:
                            vt_count = min(3, len(face_line) - 3)  # Take up to 3 indices starting from position 3
                            for j in range(vt_count):
                                try:
                                    vt_indices.append(int(face_line[j + 3]))
                                except (ValueError, IndexError):
                                    # Use a placeholder or default if conversion fails
                                    vt_indices.append(0)
                        
                        # Ensure we always have 3 texture indices (or empty list if none)
                        if vt_indices:
                            while len(vt_indices) < 3:
                                vt_indices.append(0)
                        
                        # Extract extra data
                        if len(face_line) > 6:
                            for j in range(6, len(face_line)):
                                try:
                                    extra.append(int(face_line[j]))
                                except (ValueError, IndexError):
                                    pass
                        
                        faces.append({"v": v_indices, "vt": vt_indices, "extra": extra})
                        i += 1

                    current_section["objects"].append({"texture": texture, "faces": faces})

            elif line.startswith("####"):
                i += 1
            else:
                i += 1

        if current_section is not None:
            sections.append(current_section)
        return sections
    except Exception as e:
        print(f"Error parsing ALP file: {e}")
        traceback.print_exc()  # Print the full traceback for debugging
        return []

###############################################################################
#                          ALP -> DXF CONVERSION                              #
###############################################################################

def convert_alp_to_dxf(alp_file, dxf_file):
    """Convert ALP file to DXF format"""
    sections = parse_alp_file(alp_file)
    
    # Create DXF document
    doc = ezdxf.new("R2010")
    msp = doc.modelspace()
    
    # Add a custom layer for metadata
    doc.layers.new(name="ALP_METADATA")
    
    # Add metadata about original file
    msp.add_text("ALP_FILE=" + alp_file, height=0.1).dxf.layer = "ALP_METADATA"
    
    # Process each section
    for sec_idx, section in enumerate(sections):
        if not section["verts"]:
            # Skip empty sections
            continue
            
        section_layer = f"SECTION_{sec_idx}"
        doc.layers.new(name=section_layer)
        
        # Add vertices as points for reference (optional)
        for idx, vert in enumerate(section["verts"]):
            point = msp.add_point((vert[0], vert[1], vert[2]))
            point.dxf.layer = section_layer
            # Add vertex index as extended data
            point.set_xdata("ALP_DATA", [(1000, f"VERT_IDX:{idx}")])
        
        # Process each object in the section
        for obj_idx, obj in enumerate(section["objects"]):
            obj_layer = f"{section_layer}_OBJ_{obj_idx}"
            doc.layers.new(name=obj_layer)
            
            # Store texture path as extended data for the layer
            layer = doc.layers.get(obj_layer)
            layer.set_xdata("ALP_DATA", [(1000, f"TEXTURE:{obj['texture']}")])
            
            # Draw faces as 3DFACE entities
            for face_idx, face in enumerate(obj["faces"]):
                v_indices = face["v"]
                vt_indices = face["vt"]
                extra = face["extra"]
                
                if len(v_indices) >= 3 and all(idx < len(section["verts"]) for idx in v_indices):
                    # Get the 3D vertices
                    v1 = section["verts"][v_indices[0]]
                    v2 = section["verts"][v_indices[1]]
                    v3 = section["verts"][v_indices[2]]
                    
                    # Create 3DFACE entity with vertices as a list
                    try:
                        face_entity = msp.add_3dface([
                            (v1[0], v1[1], v1[2]),
                            (v2[0], v2[1], v2[2]),
                            (v3[0], v3[1], v3[2]),
                            (v3[0], v3[1], v3[2])  # Repeat last vertex for triangular face
                        ])
                        
                        face_entity.dxf.layer = obj_layer
                        
                        # Store face data as extended data
                        xdata = [(1000, f"FACE_IDX:{face_idx}")]
                        
                        # Store vertex indices
                        v_str = ",".join(map(str, v_indices))
                        xdata.append((1000, f"V_INDICES:{v_str}"))
                        
                        # Store texture vertex indices if present
                        if vt_indices:
                            vt_str = ",".join(map(str, vt_indices))
                            xdata.append((1000, f"VT_INDICES:{vt_str}"))
                        
                        # Store extra data if present
                        if extra:
                            extra_str = ",".join(map(str, extra))
                            xdata.append((1000, f"EXTRA:{extra_str}"))
                        
                        face_entity.set_xdata("ALP_DATA", xdata)
                    except Exception as e:
                        print(f"Error creating face: {e}")
    
    # Save DXF file
    doc.saveas(dxf_file)
    return True

###############################################################################
#                          DXF -> ALP CONVERSION                              #
###############################################################################

def convert_dxf_to_alp(dxf_file, alp_file):
    """Convert DXF file back to ALP format"""
    try:
        # Load DXF file
        doc = ezdxf.readfile(dxf_file)
        
        # Extract ALP data structure from DXF
        sections = []
        
        # Get all layers and sort them by section
        section_layers = {}
        
        for layer_name in doc.layers:
            if layer_name.dxftype() != "LAYER" or layer_name.dxf.name == "ALP_METADATA":
                continue
            
            name = layer_name.dxf.name
            
            if name.startswith("SECTION_"):
                section_idx = int(name.split("_")[1])
                if section_idx not in section_layers:
                    section_layers[section_idx] = {"main": name, "objects": {}}
            
            elif "_OBJ_" in name:
                parts = name.split("_")
                if len(parts) >= 4 and parts[0] == "SECTION":
                    section_idx = int(parts[1])
                    obj_idx = int(parts[3])
                    
                    if section_idx not in section_layers:
                        section_layers[section_idx] = {"main": f"SECTION_{section_idx}", "objects": {}}
                    
                    section_layers[section_idx]["objects"][obj_idx] = name
        
        # Process each section
        for section_idx in sorted(section_layers.keys()):
            section_info = section_layers[section_idx]
            section_layer = section_info["main"]
            
            # Create a new section
            section = {"verts": [], "tverts": [], "objects": []}
            
            # Get all vertices from points in the section layer
            msp = doc.modelspace()
            vertex_map = {}  # Maps original ALP indices to new indices
            
            for entity in msp:
                if entity.dxftype() == "POINT" and entity.dxf.layer == section_layer:
                    if entity.has_xdata("ALP_DATA"):
                        xdata = entity.get_xdata("ALP_DATA")
                        vert_idx = -1
                        
                        for item in xdata:
                            if item[0] == 1000 and item[1].startswith("VERT_IDX:"):
                                vert_idx = int(item[1].split(":")[1])
                                break
                        
                        if vert_idx >= 0:
                            vertex = (entity.dxf.location[0], entity.dxf.location[1], entity.dxf.location[2])
                            section["verts"].append(vertex)
                            vertex_map[vert_idx] = len(section["verts"]) - 1
            
            # Get all texture vertices (these might be stored as extended data)
            tverts = []
            
            # Process each object in the section
            for obj_idx in sorted(section_info["objects"].keys()):
                obj_layer = section_info["objects"][obj_idx]
                layer = doc.layers.get(obj_layer)
                
                # Get texture path from layer extended data
                texture_path = ""
                if layer and layer.has_xdata("ALP_DATA"):
                    xdata = layer.get_xdata("ALP_DATA")
                    for item in xdata:
                        if item[0] == 1000 and item[1].startswith("TEXTURE:"):
                            texture_path = item[1].split(":", 1)[1]
                            break
                
                # Create a new object
                obj = {"texture": texture_path, "faces": []}
                
                # Process faces
                for entity in msp:
                    if entity.dxftype() == "3DFACE" and entity.dxf.layer == obj_layer:
                        if entity.has_xdata("ALP_DATA"):
                            xdata = entity.get_xdata("ALP_DATA")
                            v_indices = []
                            vt_indices = []
                            extra = []
                            
                            for item in xdata:
                                if item[0] == 1000:
                                    if item[1].startswith("V_INDICES:"):
                                        v_str = item[1].split(":", 1)[1]
                                        v_indices = list(map(int, v_str.split(",")))
                                    elif item[1].startswith("VT_INDICES:"):
                                        vt_str = item[1].split(":", 1)[1]
                                        vt_indices = list(map(int, vt_str.split(",")))
                                    elif item[1].startswith("EXTRA:"):
                                        extra_str = item[1].split(":", 1)[1]
                                        extra = list(map(int, extra_str.split(",")))
                            
                            # Map vertex indices using vertex_map
                            mapped_v_indices = [vertex_map.get(idx, idx) for idx in v_indices]
                            
                            # Add face to object
                            face = {"v": mapped_v_indices, "vt": vt_indices, "extra": extra}
                            obj["faces"].append(face)
                
                # Add object to section
                section["objects"].append(obj)
            
            # Add section to sections
            sections.append(section)
        
        # Write ALP file
        write_alp_file(alp_file, sections)
        return True
        
    except Exception as e:
        print(f"Error converting DXF to ALP: {e}")
        traceback.print_exc()  # Print the full traceback for debugging
        return False

###############################################################################
#                          ALP -> OBJ CONVERSION                              #
###############################################################################

def write_obj_and_mtl_file(obj_filename, sections):
    """
    Writes an OBJ + MTL file from ALP sections
    """
    try:
        base_name, ext = os.path.splitext(obj_filename)
        mtl_filename = base_name + ".mtl"
        mtl_basename = os.path.basename(mtl_filename)

        material_map = {}

        with open(obj_filename, 'w') as f_obj:
            f_obj.write("# Converted from ALP for DTR2\n")
            f_obj.write(f"mtllib {mtl_basename}\n\n")

            global_vert_offset = 0
            global_tvert_offset = 0

            for sec_index, section in enumerate(sections):
                f_obj.write(f"#----- Section {sec_index} -----\n")
                
                # Validate vertices data
                valid_verts = []
                for v in section.get("verts", []):
                    if len(v) >= 3:  # Ensure we have at least x, y, z
                        valid_verts.append(v)
                
                # Write v lines
                for v in valid_verts:
                    f_obj.write(f"v {v[0]} {v[1]} {v[2]}\n")
                
                # Validate texture vertices data
                valid_tverts = []
                for vt in section.get("tverts", []):
                    if len(vt) >= 2:  # Ensure we have at least u, v
                        valid_tverts.append(vt)
                
                # Write vt lines
                for vt in valid_tverts:
                    f_obj.write(f"vt {vt[0]} {vt[1]}\n")

                for obj_index, obj in enumerate(section.get("objects", [])):
                    # Keep the original texture path in a comment
                    original_texture = obj.get("texture", "")

                    # Derive a short name by removing everything after ".3df"
                    base_tex = original_texture.replace("\\", "/").split("/")[-1]  # just the filename
                    # If it ends with .3df, we cut at .3df
                    if base_tex.lower().endswith(".3df"):
                        # Keep everything up to .3df
                        base_tex = base_tex[:base_tex.lower().rfind(".3df") + 4]
                    # If there's no .3df in the name, we just keep the entire filename

                    object_name = base_tex.strip()
                    if not object_name:
                        # fallback if something is empty
                        object_name = f"object_{sec_index}_{obj_index}"

                    # We'll still use a unique material name so we don't get collisions
                    material_name = f"{object_name}_mat_{sec_index}_{obj_index}"
                    material_map[object_name] = {
                        "material_name": material_name,
                        "texture_path": original_texture
                    }

                    f_obj.write(f"\no {object_name}\n")
                    f_obj.write(f"# original_texture {original_texture}\n")
                    f_obj.write(f"g {object_name}\n")
                    f_obj.write(f"usemtl {material_name}\n")
                    
                    # Number of vertices and texture vertices in this section
                    vert_count = len(valid_verts)
                    tvert_count = len(valid_tverts)

                    for face in obj.get("faces", []):
                        v_indices = face.get("v", [])
                        vt_indices = face.get("vt", [])
                        extra = face.get("extra", [])
                        
                        # Skip invalid faces
                        if not v_indices:
                            continue
                            
                        # Filter out invalid vertex indices
                        valid_v_inds = []
                        for v_i in v_indices:
                            if 0 <= v_i < vert_count:
                                valid_v_inds.append(str(global_vert_offset + v_i + 1))
                            else:
                                # Use first vertex as fallback for invalid indices
                                valid_v_inds.append(str(global_vert_offset + 1))
                        
                        # Filter out invalid texture vertex indices
                        valid_vt_inds = []
                        for vt_i in vt_indices:
                            if 0 <= vt_i < tvert_count:
                                valid_vt_inds.append(str(global_tvert_offset + vt_i + 1))
                        
                        # Skip face if we don't have enough valid vertices
                        if len(valid_v_inds) < 3:
                            continue
                            
                        face_parts = []
                        for idx in range(len(valid_v_inds)):
                            vi = valid_v_inds[idx]
                            if idx < len(valid_vt_inds):
                                ti = valid_vt_inds[idx]
                                face_parts.append(f"{vi}/{ti}")
                            else:
                                face_parts.append(f"{vi}")
                        
                        if face_parts:  # Only write if we have valid face parts
                            f_obj.write("f " + " ".join(face_parts) + "\n")
                            
                            if extra:
                                f_obj.write("# EXTRA " + " ".join(map(str, extra)) + "\n")

                global_vert_offset += len(valid_verts)
                global_tvert_offset += len(valid_tverts)

        write_mtl_file(mtl_filename, material_map)
        return True
    except Exception as e:
        print(f"Error writing OBJ file: {e}")
        traceback.print_exc()  # Print the full traceback for debugging
        return False

def write_mtl_file(mtl_filename, material_map):
    try:
        with open(mtl_filename, 'w') as f_mtl:
            f_mtl.write("# MTL file for DTR2 ALP conversion\n")
            for obj_name, mat_info in material_map.items():
                mat_name = mat_info["material_name"]
                texture_path = mat_info["texture_path"]
                f_mtl.write(f"newmtl {mat_name}\n")
                f_mtl.write("Ka 1.0 1.0 1.0\n")
                f_mtl.write("Kd 1.0 1.0 1.0\n")
                f_mtl.write("Ks 0.0 0.0 0.0\n")
                f_mtl.write("d 1.0\n")
                f_mtl.write("illum 2\n")
                if texture_path and texture_path.lower() != "none":
                    f_mtl.write(f"map_Kd {texture_path}\n")
                f_mtl.write("\n")
        return True
    except Exception as e:
        print(f"Error writing MTL file: {e}")
        traceback.print_exc()  # Print the full traceback for debugging
        return False

###############################################################################
#                          OBJ -> ALP CONVERSION                              #
###############################################################################

def parse_obj_file(filename):
    try:
        with open(filename, 'r') as f:
            lines = f.read().splitlines()

        sections = []
        current_section = None
        current_object = None
        i = 0
        while i < len(lines):
            if i >= len(lines):
                break
                
            line = lines[i].strip()

            if line.startswith("#----- Section"):
                if current_section is not None:
                    if current_object is not None:
                        current_section["objects"].append(current_object)
                        current_object = None
                    sections.append(current_section)
                current_section = {"verts": [], "tverts": [], "objects": []}
                i += 1
                continue

            if line.startswith("v "):
                parts = line.split()[1:]
                if current_section is None:
                    current_section = {"verts": [], "tverts": [], "objects": []}
                try:
                    if len(parts) >= 3:
                        v = list(map(float, parts[:3]))  # Take only x, y, z
                        current_section["verts"].append(v)
                except ValueError:
                    # Skip invalid vertex data
                    pass
                i += 1
                continue

            if line.startswith("vt "):
                parts = line.split()[1:]
                if current_section is None:
                    current_section = {"verts": [], "tverts": [], "objects": []}
                try:
                    if len(parts) >= 2:
                        vt = list(map(float, parts[:2]))  # Take only u, v
                        current_section["tverts"].append(vt)
                except ValueError:
                    # Skip invalid texture vertex data
                    pass
                i += 1
                continue

            if line.startswith("o "):
                if current_object is not None and current_section is not None:
                    current_section["objects"].append(current_object)
                object_name = line[2:].strip()
                current_object = {"texture": object_name, "faces": []}
                i += 1
                continue

            if line.startswith("# original_texture"):
                try:
                    real_tex = line.split(" ", 2)[2].strip()  # everything after '# original_texture '
                    if current_object is not None:
                        current_object["texture"] = real_tex
                except IndexError:
                    # Skip if format is invalid
                    pass
                i += 1
                continue

            if line.startswith("g ") or line.startswith("usemtl "):
                i += 1
                continue

            if line.startswith("f "):
                if current_section is None:
                    current_section = {"verts": [], "tverts": [], "objects": []}
                if current_object is None:
                    current_object = {"texture": "", "faces": []}
                    
                parts = line.split()[1:]
                v_indices = []
                vt_indices = []
                
                for part in parts:
                    if "/" in part:
                        sp = part.split("/")
                        try:
                            v_i = int(sp[0]) - 1
                            v_indices.append(v_i)
                            
                            if len(sp) > 1 and sp[1] != "":
                                vt_i = int(sp[1]) - 1
                                vt_indices.append(vt_i)
                        except (ValueError, IndexError):
                            # Skip invalid indices
                            continue
                    else:
                        try:
                            v_indices.append(int(part) - 1)
                        except (ValueError, IndexError):
                            # Skip invalid indices
                            continue
                
                # Only process face if we have valid vertex indices
                if len(v_indices) >= 3:
                    extra = []
                    if (i+1) < len(lines):
                        nxt = lines[i+1].strip()
                        if nxt.startswith("# EXTRA"):
                            try:
                                extra_parts = nxt.split()[2:]
                                extra = list(map(int, extra_parts))
                                i += 1
                            except (ValueError, IndexError):
                                # Skip invalid extra data
                                pass

                    # Ensure we have 3 vertex indices (triangular faces)
                    v_indices = v_indices[:3]
                    
                    # Match length of vt_indices with v_indices by filling zeros
                    if vt_indices:
                        while len(vt_indices) < len(v_indices):
                            vt_indices.append(0)
                        vt_indices = vt_indices[:3]  # Limit to 3 indices

                    face = {"v": v_indices, "vt": vt_indices, "extra": extra}
                    current_object["faces"].append(face)
                
                i += 1
                continue

            i += 1

        if current_object is not None and current_section is not None:
            current_section["objects"].append(current_object)
        if current_section is not None:
            sections.append(current_section)

        return sections
    except Exception as e:
        print(f"Error parsing OBJ file: {e}")
        traceback.print_exc()  # Print the full traceback for debugging
        return []

def write_alp_file(filename, sections):
    try:
        with open(filename, 'w') as f:
            # Write number of meshes (sections) header
            f.write(f"num meshes {len(sections)}\n\n")
            
            for sec_idx, section in enumerate(sections):
                # Skip empty sections
                if not section.get("verts"):
                    continue
                    
                f.write(f"num verts {len(section['verts'])}\n")
                for v in section["verts"]:
                    # Ensure we have 3 coordinates
                    if len(v) >= 3:
                        f.write(f"{v[0]} {v[1]} {v[2]}\n")
                    else:
                        # Use zeros for missing coordinates
                        coords = v + [0] * (3 - len(v))
                        f.write(f"{coords[0]} {coords[1]} {coords[2]}\n")

                f.write(f"\nnum tverts {len(section.get('tverts', []))}\n")
                for vt in section.get("tverts", []):
                    # Ensure we have at least 2 texture coordinates
                    if len(vt) >= 2:
                        f.write(f"\t\t{vt[0]} {vt[1]}\n")
                    else:
                        # Use zeros for missing coordinates
                        coords = vt + [0] * (2 - len(vt))
                        f.write(f"\t\t{coords[0]} {coords[1]}\n")

                objects = section.get("objects", [])
                f.write(f"\tnum objects {len(objects)}\n")
                for obj in objects:
                    texture = obj.get("texture", "")
                    f.write(f"\t\ttexture {texture}\n")
                    
                    faces = obj.get("faces", [])
                    f.write(f"\t\t\tnum faces {len(faces)}\n")
                    
                    for face in faces:
                        v_part = face.get("v", [])
                        vt_part = face.get("vt", [])
                        extra_part = face.get("extra", [])
                        
                        # Ensure we have valid data
                        if not v_part:
                            v_part = [0, 0, 0]
                        while len(v_part) < 3:
                            v_part.append(0)
                        
                        if vt_part:
                            while len(vt_part) < 3:
                                vt_part.append(0)
                        
                        line_nums = v_part[:3]
                        if vt_part:
                            line_nums.extend(vt_part[:3])
                        if extra_part:
                            line_nums.extend(extra_part)
                            
                        str_line = "\t\t\t" + " ".join(map(str, line_nums))
                        f.write(f"{str_line}\n")

                if sec_idx < len(sections) - 1:
                    f.write("\n####\n\n")
        return True
    except Exception as e:
        print(f"Error writing ALP file: {e}")
        traceback.print_exc()  # Print the full traceback for debugging
        return False

###############################################################################
#                               TKINTER GUI                                   #
###############################################################################

class ALPConverterGUI:
    def __init__(self, root):
        self.root = root
        self.root.title("DTR2 ALP File Converter")
        self.root.geometry("600x500")
        
        # Create main frame
        main_frame = ttk.Frame(root, padding=10)
        main_frame.pack(fill=tk.BOTH, expand=True)
        
        # Conversion mode frame
        mode_frame = ttk.LabelFrame(main_frame, text="Conversion Mode", padding=10)
        mode_frame.pack(fill=tk.X, pady=(0, 10))
        
        self.mode_var = tk.StringVar(value="alp2dxf")
        
        # Mode radio buttons
        modes = [
            ("ALP to DXF (for Rhino)", "alp2dxf"),
            ("ALP to OBJ (for other 3D software)", "alp2obj"),
            ("DXF to ALP (after Rhino editing)", "dxf2alp"),
            ("OBJ to ALP (after editing)", "obj2alp")
        ]
        
        for i, (text, value) in enumerate(modes):
            ttk.Radiobutton(mode_frame, text=text, value=value, variable=self.mode_var).grid(
                row=i//2, column=i%2, sticky=tk.W, padx=5, pady=2
            )
        
        # File selection frame
        file_frame = ttk.LabelFrame(main_frame, text="File Selection", padding=10)
        file_frame.pack(fill=tk.X, pady=(0, 10))
        
        # Input file
        ttk.Label(file_frame, text="Input File:").grid(row=0, column=0, sticky=tk.W, pady=5)
        self.input_var = tk.StringVar()
        input_entry = ttk.Entry(file_frame, textvariable=self.input_var, width=50)
        input_entry.grid(row=0, column=1, padx=5, pady=5)
        ttk.Button(file_frame, text="Browse...", command=self.browse_input).grid(row=0, column=2, padx=5, pady=5)
        
        # Output file
        ttk.Label(file_frame, text="Output File:").grid(row=1, column=0, sticky=tk.W, pady=5)
        self.output_var = tk.StringVar()
        output_entry = ttk.Entry(file_frame, textvariable=self.output_var, width=50)
        output_entry.grid(row=1, column=1, padx=5, pady=5)
        ttk.Button(file_frame, text="Browse...", command=self.browse_output).grid(row=1, column=2, padx=5, pady=5)
        
        # Batch conversion option
        batch_frame = ttk.Frame(file_frame)
        batch_frame.grid(row=2, column=0, columnspan=3, sticky=tk.W, pady=5)
        
        self.batch_var = tk.BooleanVar(value=False)
        ttk.Checkbutton(batch_frame, text="Batch Mode: Convert all files in folder", variable=self.batch_var).pack(side=tk.LEFT)
        ttk.Label(batch_frame, text="(Input should be a folder in batch mode)").pack(side=tk.LEFT, padx=5)
        
        # Conversion button
        button_frame = ttk.Frame(main_frame)
        button_frame.pack(fill=tk.X, pady=10)
        
        ttk.Button(button_frame, text="Convert Now", command=self.convert).pack(side=tk.LEFT, padx=5)
        ttk.Button(button_frame, text="Exit", command=root.destroy).pack(side=tk.RIGHT, padx=5)
        
        # Log display
        log_frame = ttk.LabelFrame(main_frame, text="Status Log", padding=10)
        log_frame.pack(fill=tk.BOTH, expand=True, pady=(0, 10))
        
        self.log_text = tk.Text(log_frame, wrap=tk.WORD, width=70, height=10)
        self.log_text.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        
        scrollbar = ttk.Scrollbar(log_frame, orient=tk.VERTICAL, command=self.log_text.yview)
        scrollbar.pack(side=tk.RIGHT, fill=tk.Y)
        self.log_text.config(yscrollcommand=scrollbar.set)
        
        # Initialize
        self.log("Ready to convert files. Please select conversion mode and files.")
    
    def log(self, message):
        """Add a message to the log display"""
        self.log_text.insert(tk.END, message + "\n")
        self.log_text.see(tk.END)
        self.root.update_idletasks()
    
    def browse_input(self):
        """Browse for input file or directory"""
        mode = self.mode_var.get()
        batch = self.batch_var.get()
        
        if batch:
            # Browse for directory
            directory = filedialog.askdirectory(title="Select Input Directory")
            if directory:
                self.input_var.set(directory)
                # Suggest output directory
                if not self.output_var.get():
                    self.output_var.set(directory)
        else:
            # Browse for file
            if mode.startswith("alp"):
                filetypes = [("ALP Files", "*.alp"), ("All Files", "*.*")]
            elif mode.startswith("dxf"):
                filetypes = [("DXF Files", "*.dxf"), ("All Files", "*.*")]
            else:  # obj2alp
                filetypes = [("OBJ Files", "*.obj"), ("All Files", "*.*")]
                
            file = filedialog.askopenfilename(title="Select Input File", filetypes=filetypes)
            if file:
                self.input_var.set(file)
                # Suggest output filename
                if not self.output_var.get():
                    base, ext = os.path.splitext(file)
                    if mode == "alp2dxf":
                        self.output_var.set(base + ".dxf")
                    elif mode == "alp2obj":
                        self.output_var.set(base + ".obj")
                    elif mode == "dxf2alp":
                        self.output_var.set(base + ".alp")
                    elif mode == "obj2alp":
                        self.output_var.set(base + ".alp")
    
    def browse_output(self):
        """Browse for output file or directory"""
        mode = self.mode_var.get()
        batch = self.batch_var.get()
        
        if batch:
            # Browse for directory
            directory = filedialog.askdirectory(title="Select Output Directory")
            if directory:
                self.output_var.set(directory)
        else:
            # Browse for file
            if mode.endswith("2dxf"):
                filetypes = [("DXF Files", "*.dxf"), ("All Files", "*.*")]
                defaultext = ".dxf"
            elif mode.endswith("2obj"):
                filetypes = [("OBJ Files", "*.obj"), ("All Files", "*.*")]
                defaultext = ".obj"
            else:  # to alp
                filetypes = [("ALP Files", "*.alp"), ("All Files", "*.*")]
                defaultext = ".alp"
                
            file = filedialog.asksaveasfilename(
                title="Select Output File", 
                filetypes=filetypes,
                defaultextension=defaultext
            )
            if file:
                self.output_var.set(file)
    
    def convert(self):
        """Perform the conversion"""
        mode = self.mode_var.get()
        batch = self.batch_var.get()
        input_path = self.input_var.get()
        output_path = self.output_var.get()
        
        if not input_path:
            messagebox.showerror("Error", "Please select an input file or directory.")
            return
        
        if not output_path:
            messagebox.showerror("Error", "Please select an output file or directory.")
            return
        
        if batch:
            # Batch conversion
            if not os.path.isdir(input_path):
                messagebox.showerror("Error", "Input path must be a directory in batch mode.")
                return
            
            if not os.path.isdir(output_path):
                messagebox.showerror("Error", "Output path must be a directory in batch mode.")
                return
            
            # Find all files of appropriate type
            if mode.startswith("alp"):
                files = [f for f in os.listdir(input_path) if f.lower().endswith(".alp")]
                ext = ".alp"
            elif mode.startswith("dxf"):
                files = [f for f in os.listdir(input_path) if f.lower().endswith(".dxf")]
                ext = ".dxf"
            else:  # obj2alp
                files = [f for f in os.listdir(input_path) if f.lower().endswith(".obj")]
                ext = ".obj"
            
            if not files:
                messagebox.showerror("Error", f"No {ext} files found in the input directory.")
                return
            
            self.log(f"Found {len(files)} {ext} files to convert.")
            
            success_count = 0
            error_count = 0
            
            for filename in files:
                input_file = os.path.join(input_path, filename)
                base, _ = os.path.splitext(filename)
                
                if mode == "alp2dxf":
                    output_file = os.path.join(output_path, base + ".dxf")
                    result = self.convert_single_file("alp2dxf", input_file, output_file)
                elif mode == "alp2obj":
                    output_file = os.path.join(output_path, base + ".obj")
                    result = self.convert_single_file("alp2obj", input_file, output_file)
                elif mode == "dxf2alp":
                    output_file = os.path.join(output_path, base + ".alp")
                    result = self.convert_single_file("dxf2alp", input_file, output_file)
                else:  # obj2alp
                    output_file = os.path.join(output_path, base + ".alp")
                    result = self.convert_single_file("obj2alp", input_file, output_file)
                
                if result:
                    success_count += 1
                else:
                    error_count += 1
            
            self.log(f"Conversion complete: {success_count} succeeded, {error_count} failed.")
            messagebox.showinfo("Batch Conversion", 
                               f"Conversion complete:\n{success_count} succeeded\n{error_count} failed")
            
        else:
            # Single file conversion
            if not os.path.isfile(input_path):
                messagebox.showerror("Error", "Input file does not exist.")
                return
            
            result = self.convert_single_file(mode, input_path, output_path)
            
            if result:
                self.log("Conversion completed successfully!")
                messagebox.showinfo("Success", "File converted successfully!")
            else:
                messagebox.showerror("Error", "Conversion failed. Check the log for details.")
    
    def convert_single_file(self, mode, input_file, output_file):
        """Convert a single file"""
        self.log(f"Converting {os.path.basename(input_file)} -> {os.path.basename(output_file)}")
        
        try:
            if mode == "alp2dxf":
                # ALP to DXF
                success = convert_alp_to_dxf(input_file, output_file)
            elif mode == "alp2obj":
                # ALP to OBJ
                sections = parse_alp_file(input_file)
                if not sections:
                    self.log("Error: No valid sections found in ALP file.")
                    return False
                success = write_obj_and_mtl_file(output_file, sections)
            elif mode == "dxf2alp":
                # DXF to ALP
                success = convert_dxf_to_alp(input_file, output_file)
            else:  # obj2alp
                # OBJ to ALP
                sections = parse_obj_file(input_file)
                if not sections:
                    self.log("Error: No valid sections found in OBJ file.")
                    return False
                success = write_alp_file(output_file, sections)
            
            if success:
                self.log(f"Successfully converted {os.path.basename(input_file)}.")
                return True
            else:
                self.log(f"Failed to convert {os.path.basename(input_file)}.")
                return False
                
        except Exception as e:
            self.log(f"Error: {str(e)}")
            traceback.print_exc()  # Print the full traceback for debugging
            return False

# Run the application
if __name__ == "__main__":
    root = tk.Tk()
    app = ALPConverterGUI(root)
    root.mainloop()